<?php

namespace App\Http\Controllers;

use Illuminate\Http\JsonResponse;

class ApiController extends Controller
{

    /**
     * Respond the data.
     *
     * @param array $array
     * @param int $statusCode
     * @return JsonResponse
     */
    public function respondWithArray(array $array, $statusCode=200)
    {
        return response()->json($array, $statusCode);
    }

    /**
     * Respond the message.
     *
     * @param string $message
     * @param int $statusCode
     * @return JsonResponse
     */

    public function respondWithMessage($message, $statusCode)
    {
        if ($statusCode == 200) {
            $status = 'success';
        } else if (in_array($statusCode, [401, 403, 404, 419, 422])) {
            $status = 'failed';
        } else {
            $status = 'error';
        }

        return $this->respondWithArray([
            'status' => $status,
            'message' => $message,
        ], $statusCode);
    }

    /**
     *
     * @param $message
     * @return JsonResponse
     */

    function respondWithSuccess($message){
        return $this->respondWithMessage($message, 200);
    }

    /**
     * Respond the error message.
     *
     * @param string $message
     * @param int $statusCode
     * @return JsonResponse
     */
    protected function respondWithError($message, $statusCode = 500)
    {
        return $this->respondWithMessage($message, $statusCode);
    }

    /**
     * Respond the error of 'Forbidden'
     *
     * @param string $message
     * @return JsonResponse
     */
    public function forbidden($message = 'You don’t have permission to access')
    {
        return $this->respondWithMessage($message, 403);
    }

    /**
     * Respond the error of 'Resource Not Found'
     *
     * @param string $message
     * @return JsonResponse
     */
    public function notFound($message = 'Resource Not Found')
    {
        return $this->respondWithMessage($message, 404);
    }

    /**
     * Respond the error of 'Unauthorized'.
     *
     * @param string $message
     * @return JsonResponse
     */
    public function unauthorized($message = 'Unauthorized')
    {
        return $this->respondWithMessage($message, 401);
    }

    /**
     * @param array $errors
     * @return JsonResponse
     */

    public function validationError(array $errors)
    {
        $message = implode(',', $errors);
        return $this->respondWithMessage($message, 422);
    }

    /**
     *
     * @param \Exception $exception
     * @return JsonResponse
     */

    public function handleException($exception){
      \Log::emergency('File: ' . $exception->getFile() .' Line: ' . $exception->getLine() . ' Message: '. $exception->getMessage());
      return $this->respondWithError($exception->getMessage());
    }

    public function respondWithData($data){
        return $this->respondWithArray([
            'status' => 'success',
            'data' => $data,
        ], 200);
    }

}
