import { isEmpty, isEmptyArray, isNullOrUndefined } from '@core/utils'
import {firstLetterUpperCase} from "@/utils/common"

// 👉 Required Validator
export const required = (value, params, { field }) => {

  let message = `${firstLetterUpperCase(field)} is required`
  if (isNullOrUndefined(value) || isEmptyArray(value) || value === false)
    return message

  return !!String(value).trim().length || message
}

// 👉 Email Validator
export const email = (value, params, inputField) => {
  if (isEmpty(value))
    return true
  const re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/
  if (Array.isArray(value))
    return value.every(val => re.test(String(val))) || 'The Email field must be a valid email'

  return re.test(String(value)) || 'The Email field must be a valid email'
}

// 👉 Password Validator
export const password = password => {
  const regExp = /(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[!@#$%&*()]).{8,}/
  const validPassword = regExp.test(password)

  return (
    // eslint-disable-next-line operator-linebreak
    validPassword ||
    'Field must contain at least one uppercase, lowercase, special character and digit with min 8 chars')
}

// 👉 Confirm Password Validator
export const confirmed = (value, target) => value === target || 'The Confirm Password field confirmation does not match'

// 👉 Between Validator
export const betweenNumber = (value, params, { field }) => {

  const valueAsNumber = Number(value)
  const min = params[0]
  const max = params[1]
  return (Number(min) <= valueAsNumber && Number(max) >= valueAsNumber) || `Enter number between ${min} and ${max}`
}

//between number of characters

export const between = (value, params, { field }) => {

  const min = params[0]
  const max = params[1]

  if (isEmpty(value)) {
    return true
  }
  return Number(min) <= value.length && value.length <= Number(max) || `${firstLetterUpperCase(field)} must be ${min}-${max} characters long`

}

//between number of characters

export const min = (value, params, { field }) => {

  const min = params[0]

  if (isEmpty(value)) {
    return true
  }
  return Number(min) <= value.length || `${firstLetterUpperCase(field)} must be at least ${min} characters long`

}

//between number of characters

export const max = (value, params, { field }) => {

  const max = params[0]

  if (isEmpty(value)) {
    return true
  }
  return value.length <= Number(max) || `${firstLetterUpperCase(field)} must be ${max} characters long`

}

// 👉 Integer Validator
export const integer = value => {
  if (isEmpty(value))
    return true
  if (Array.isArray(value))
    return value.every(val => /^-?[0-9]+$/.test(String(val))) || 'This field must be an integer'

  return /^-?[0-9]+$/.test(String(value)) || 'This field must be an integer'
}

// 👉 Regex Validator
export const regex = (value, params, { field }) => {
  if (isEmpty(value))
    return true
  let regex = new RegExp(params[0])

  return regex.test(String(value)) || `${firstLetterUpperCase(field)} is invalid`
}

//BD Phone validator

export const phoneBd = (value, params, { field }) => {
  if (isEmpty(value)) {
    return true
  }
  const regex = /(^([+]8{2}|0088)?(01)[3-9]\d{8})$/
  return regex.test(String(value)) || `${firstLetterUpperCase(field)} is not valid`
}


//Username Validator

export const username = (value, params, { field }) => {
  if (isEmpty(value)) {
    return true
  }
  const regex = /^[a-zA-Z]\w+$/
  return regex.test(String(value)) || `${firstLetterUpperCase(field)} must be started with an alphabet and can contain only alphabets, digits and underscore`
}

//Human Name validator

export const nameWithSpace = (value, params, { field }) => {
  if (isEmpty(value)) {
    return true
  }
  const regex = /^[a-zA-Z][a-zA-Z ]+/
  return regex.test(String(value)) || `${field.charAt(0).toUpperCase() + field.slice(1)} must be started with alphabet and can contain only alphabet and white space`
}

//Name with Number

export const nameWithNumber = (value, params, { field }) => {
  if (isEmpty(value)) {
    return true
  }
  const regex = /^[a-zA-Z][\w ]+$/
  return regex.test(value) || `${firstLetterUpperCase(field)} must be started with an alphabet and can contain only alphabets, digits and whitespace`
}

//Number Validator

export const number = (value, params, { field }) => {
  if (isEmpty(value)) {
    return true
  }

  return !isNaN(value) || `${firstLetterUpperCase(field)} is not a valid number`
}

// 👉 Alpha Validator
export const alpha = value => {
  if (isEmpty(value))
    return true

  return /^[A-Z]*$/i.test(String(value)) || 'The Alpha field may only contain alphabetic characters'
}

// 👉 URL Validator
export const url = value => {
  if (isEmpty(value))
    return true
  const re = /^(http[s]?:\/\/){0,1}(www\.){0,1}[a-zA-Z0-9\.\-]+\.[a-zA-Z]{2,5}[\.]{0,1}/

  return re.test(String(value)) || 'URL is invalid'
}

// 👉 Length Validator
export const length = (value, length) => {
  if (isEmpty(value))
    return true

  return String(value).length === length || `The Min Character field must be at least ${length} characters`
}

// 👉 Alpha-dash Validator
export const alphaDash = value => {
  if (isEmpty(value))
    return true
  const valueAsString = String(value)

  return /^[0-9A-Z_-]*$/i.test(valueAsString) || 'All Character are not valid'
}

export const alphaNumeric = (value,params,{field}) => {
  if(isEmpty(value)){
    return true;
  }
  return /^[\dA-Z]*$/i.test(String(value)) || `${field} can contain only alphabet and numbers`
}
